"""Meter utilities for the present‑act engine.

The which‑path meter emits symbols drawn from a finite alphabet of
size ``M``.  For the simplified Simulation A the meter feature sets
are computed analytically based on the overlap parameter ``m``.  The
resulting distinguishability can also be derived in closed form.
"""

from __future__ import annotations
from typing import Tuple, Set


def make_feature_sets(M: int, m: int) -> Tuple[Set[int], Set[int]]:
    """Construct symmetric feature sets for the meter.

    For the upper path (U) and lower path (L) we define two subsets
    ``F_U`` and ``F_L`` of ``{0, 1, …, M-1}`` with overlap size ``m``.
    The overlap comprises the first ``m`` symbols; the remaining ``M-m``
    symbols are divided evenly between the two sets.

    Parameters
    ----------
    M : int
        Total number of meter symbols.
    m : int
        Size of the overlap between ``F_U`` and ``F_L``.  Must satisfy
        ``0 <= m <= M`` and ``(M - m)`` must be even.

    Returns
    -------
    (F_U, F_L) : pair of sets
        Feature sets for the upper and lower paths.
    """
    assert 0 <= m <= M, "m must be between 0 and M"
    remaining = M - m
    assert remaining % 2 == 0, "M - m must be even"
    half = remaining // 2
    base = list(range(M))
    # Overlap region
    overlap = base[:m]
    # Unique parts for U and L
    unique_u = base[m : m + half]
    unique_l = base[m + half :]
    F_U = set(overlap + unique_u)
    F_L = set(overlap + unique_l)
    return F_U, F_L